jQuery(document).ready(function($) {
    
    // Initialize ICP Ajax object if not already available (fallback)
    if (typeof icp_ajax === 'undefined') {
        console.log('ICP: Ajax object not found, checking if we can initialize it...');
        // This might happen in media modal context, try to get nonces from WordPress
        if (typeof ajaxurl !== 'undefined') {
            window.icp_ajax = {
                ajax_url: ajaxurl,
                compress_single_nonce: $('#_wpnonce').val() || 'fallback'
            };
            console.log('ICP: Fallback ajax object created');
        }
    }
    
    // Bulk Compression Functionality
    let bulkCompressionRunning = false;
    let bulkCompressionStopped = false;
    let totalImages = 0;
    let processedImages = 0;
    let successfulCompressions = 0;
    let totalErrors = 0;
    let sessionSuccessfulCompressions = 0; // Track only current session compressions
    let sessionErrors = 0; // Track only current session errors
    
    $('#icp-start-bulk-compression').on('click', function() {
        if (bulkCompressionRunning) return;
        
        bulkCompressionRunning = true;
        bulkCompressionStopped = false;
        processedImages = 0;
        successfulCompressions = 0;
        totalErrors = 0;
        sessionSuccessfulCompressions = 0;
        sessionErrors = 0;
        
        $(this).hide();
        $('#icp-retry-failed-only').hide();
        $('#icp-stop-bulk-compression').show();
        $('#icp-bulk-progress').show();
        $('#icp-bulk-results').hide();
        $('#icp-results-content').empty(); // Clear previous results
        
        startBulkCompression(0, false);
    });
    
    $('#icp-retry-failed-only').on('click', function() {
        if (bulkCompressionRunning) return;
        
        bulkCompressionRunning = true;
        bulkCompressionStopped = false;
        processedImages = 0;
        successfulCompressions = 0;
        totalErrors = 0;
        sessionSuccessfulCompressions = 0;
        sessionErrors = 0;
        
        $(this).hide();
        $('#icp-start-bulk-compression').hide();
        $('#icp-stop-bulk-compression').show();
        $('#icp-bulk-progress').show();
        $('#icp-bulk-results').hide();
        $('#icp-results-content').empty(); // Clear previous results
        
        startBulkCompression(0, true);
    });
    
    $('#icp-stop-bulk-compression').on('click', function() {
        bulkCompressionStopped = true;
        bulkCompressionRunning = false;
        
        $(this).hide();
        $('#icp-start-bulk-compression').show();
        $('#icp-retry-failed-only').show();
        
        $('#icp-results-content').append('<div class="icp-result-error">Bulk compression stopped by user.</div>');
        $('#icp-bulk-results').show();
    });
    
    function startBulkCompression(offset, failedOnly) {
        if (bulkCompressionStopped) {
            console.log('Bulk compression stopped by user');
            return;
        }
        
        // Show current processing status
        updateProgressText('Processing batch starting at image ' + (offset + 1) + '...');
        
        console.log('Starting bulk compression at offset:', offset, 'failed only:', failedOnly || false);
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_bulk_compress',
                nonce: icp_ajax.bulk_compress_nonce,
                offset: offset,
                failed_only: failedOnly || false
            },
            success: function(response) {
                console.log('Bulk compression response:', response); // Debug log
                
                if (response.success) {
                    // Update total images count on first batch
                    if (offset === 0) {
                        totalImages = response.total;
                        $('#icp-progress-total').text(totalImages);
                        console.log('Total images to process:', totalImages);
                    }
                    
                    // Track session-specific compressions (only current session)
                    sessionSuccessfulCompressions += response.successful_in_batch;
                    sessionErrors += response.batch_errors;

                    // Keep total database counts for reference but use session counts for display
                    totalErrors = response.total_errors_so_far;
                    successfulCompressions = response.total_compressed_so_far;

                    // Update processed count (this is just for tracking batches)
                    processedImages = response.offset;
                    
                    console.log('Progress update:', {
                        processedImages: processedImages,
                        totalImages: totalImages,
                        sessionSuccessfulCompressions: sessionSuccessfulCompressions,
                        sessionErrors: sessionErrors,
                        completed: response.completed
                    });

                    // Update progress based on actual position in the list (use session counts for display)
                    updateProgress(processedImages, totalImages, sessionSuccessfulCompressions, sessionErrors);
                    
                    // Display any errors from this batch
                    if (response.errors && response.errors.length > 0) {
                        $.each(response.errors, function(index, error) {
                            $('#icp-results-content').append(
                                '<div class="icp-result-error">Error compressing image #' + 
                                error.id + ' (' + error.title + '): ' + error.error + '</div>'
                            );
                        });
                        $('#icp-bulk-results').show();
                    }
                    
                    // Check if we need to continue or if we're done
                    console.log('Checking continuation:', {
                        completed: response.completed,
                        bulkCompressionStopped: bulkCompressionStopped,
                        nextOffset: response.offset
                    });
                    
                    if (!response.completed && !bulkCompressionStopped) {
                        // Continue with next batch after a small delay
                        console.log('Continuing to next batch at offset:', response.offset);
                        updateProgressText('Processed ' + processedImages + ' of ' + totalImages + ' images. ' + sessionSuccessfulCompressions + ' successful, ' + sessionErrors + ' failed. Continuing...');
                        
                        setTimeout(function() {
                            console.log('Timeout completed, calling next batch...');
                            startBulkCompression(response.offset, failedOnly);
                        }, 500); // Reduced delay for faster processing
                    } else {
                        // Compression completed - use session-specific counts
                        console.log('Bulk compression completed:', {
                            successful: sessionSuccessfulCompressions,
                            failed: sessionErrors,
                            reason: response.completed ? 'All images processed' : 'Stopped by user'
                        });
                        finishBulkCompression(sessionSuccessfulCompressions, sessionErrors);
                    }
                } else {
                    console.error('Bulk compression error:', response);
                    
                    // Handle API key missing error specifically
                    if (response.data && response.data.error_type === 'missing_api_key') {
                        showApiKeyError(response.data.message, response.data.settings_url, 'bulk');
                    } else {
                        showBulkError('Bulk compression failed: ' + (response.data.message || response.data || 'Unknown error'));
                    }
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error:', xhr, status, error);
                console.error('Full XHR object:', xhr);
                showBulkError('Network error occurred. Please check your connection and try again. Error: ' + error);
            },
            timeout: 30000 // 30 second timeout per batch
        });
    }
    
    function updateProgress(current, total, successful, failed) {
        if (total > 0) {
            const percentage = Math.round((current / total) * 100);
            $('.icp-progress-fill').css('width', percentage + '%');
            $('#icp-progress-current').text(current);
            $('#icp-progress-total').text(total);
            
            // Update progress text with actual success/failure counts
            updateProgressText('Processed ' + current + ' of ' + total + ' images (' + percentage + '%) - ' + successful + ' successful, ' + failed + ' failed');
        }
    }
    
    function updateProgressText(text) {
        $('.icp-progress-text').html('<strong>' + text + '</strong>');
    }
    
    function finishBulkCompression(successful, failed) {
        bulkCompressionRunning = false;
        
        $('#icp-stop-bulk-compression').hide();
        $('#icp-start-bulk-compression').show();
        $('#icp-retry-failed-only').show();
        
        let message = '<div class="icp-result-success"><strong>✅ Bulk compression completed!</strong></div>';
        message += '<div class="icp-result-success">Successfully compressed: <strong>' + successful + '</strong> images</div>';
        
        if (failed > 0) {
            message += '<div class="icp-result-error">Failed to compress: <strong>' + failed + '</strong> images (see errors above)</div>';
            message += '<div class="icp-result-info">💡 <strong>Tip:</strong> You can use the "Retry Failed Only" button to retry just the failed images, or refresh the page to see updated counts.</div>';
        }
        
        $('#icp-results-content').prepend(message);
        $('#icp-bulk-results').show();
        
        // Update final progress bar to 100%
        $('.icp-progress-fill').css('width', '100%');
        updateProgressText('✅ Compression completed! ' + successful + ' images successfully compressed, ' + failed + ' failed.');
        
        // Show completion notification
        if (typeof wp !== 'undefined' && wp.notices) {
            wp.notices.create({
                type: 'success',
                message: 'Bulk image compression completed! ' + successful + ' images compressed successfully.',
                isDismissible: true
            });
        }
        
        // Refresh the page after a short delay to show updated stats
        setTimeout(function() {
            if (confirm('Bulk compression completed!\n\nSuccessfully compressed: ' + successful + ' images\nFailed: ' + failed + ' images\n\nWould you like to refresh the page to see updated statistics?')) {
                location.reload();
            }
        }, 3000);
    }
    
    function showBulkError(message) {
        bulkCompressionRunning = false;
        
        $('#icp-stop-bulk-compression').hide();
        $('#icp-start-bulk-compression').show();
        
        $('#icp-results-content').append('<div class="icp-result-error"><strong>❌ ' + message + '</strong></div>');
        $('#icp-bulk-results').show();
        
        updateProgressText('❌ Error occurred during compression.');
    }
    
    function showApiKeyError(message, settingsUrl, context) {
        const errorHtml = '<div class="icp-api-key-error">' +
                         '<div class="icp-error-icon">🔑</div>' +
                         '<div class="icp-error-content">' +
                         '<h3>TinyPNG API Key Required</h3>' +
                         '<p>' + message + '</p>' +
                         '<p>Please configure your TinyPNG API key to enable image compression.</p>' +
                         '<div class="icp-error-actions">' +
                         '<a href="' + settingsUrl + '" class="button button-primary">Go to Settings</a>' +
                         '<button type="button" class="button button-secondary icp-dismiss-error">Dismiss</button>' +
                         '</div>' +
                         '</div>' +
                         '</div>';
        
        if (context === 'bulk') {
            bulkCompressionRunning = false;
            $('#icp-stop-bulk-compression').hide();
            $('#icp-start-bulk-compression').show();
            $('#icp-results-content').append(errorHtml);
            $('#icp-bulk-results').show();
            updateProgressText('❌ API key required for compression.');
        } else {
            // For single compression, show as a modal-like overlay
            if ($('.icp-api-key-modal').length === 0) {
                $('body').append('<div class="icp-api-key-modal"><div class="icp-modal-content">' + errorHtml + '</div></div>');
            }
        }
    }
    
    function showNotification(message, type) {
        type = type || 'info';
        const notificationClass = 'icp-notification icp-notification-' + type;
        const notification = $('<div class="' + notificationClass + '">' + message + '<button type="button" class="icp-notification-close">×</button></div>');
        
        $('body').append(notification);
        
        // Auto-hide after 5 seconds
        setTimeout(function() {
            notification.fadeOut(300, function() {
                notification.remove();
            });
        }, 5000);
    }
    
    // Handle dismiss buttons and modal close
    $(document).on('click', '.icp-dismiss-error, .icp-notification-close', function() {
        $(this).closest('.icp-api-key-error, .icp-api-key-modal, .icp-notification').fadeOut(300, function() {
            $(this).remove();
        });
    });
    
    // Close modal when clicking outside
    $(document).on('click', '.icp-api-key-modal', function(e) {
        if (e.target === this) {
            $(this).fadeOut(300, function() {
                $(this).remove();
            });
        }
    });
    
    // Single Image Compression (from Media Library)
    $(document).on('click', '.icp-compress-single', function(e) {
        e.preventDefault();
        
        const $link = $(this);
        const attachmentId = $link.data('attachment-id');
        const originalText = $link.text();
        
        $link.text('Compressing...').addClass('icp-loading');
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_compress_single',
                nonce: icp_ajax.compress_single_nonce,
                attachment_id: attachmentId
            },
            success: function(response) {
                if (response.success) {
                    // Show detailed compression info in the same format as PHP renders it
                    const detailsHtml = '<div class="icp-compression-details">' +
                                      '<span class="icp-compressed">✓ ' + response.data.savings_percent + '% saved</span>' +
                                      '<div class="icp-compression-info">' +
                                      '<span class="icp-size-info">Original: ' + response.data.original_size + '</span>' +
                                      '<span class="icp-size-info">Compressed: ' + response.data.compressed_size + '</span>' +
                                      '<span class="icp-size-info">Saved: ' + response.data.savings_bytes + '</span>' +
                                      '</div>' +
                                      '</div>';
                    
                    // Replace the compress link with the compression details
                    $link.closest('td').html(detailsHtml);
                    
                    // Show success message
                    showNotification('Image compressed successfully! Saved ' + response.data.savings_percent + '% (' + response.data.savings_bytes + ')', 'success');
                } else {
                    $link.text(originalText).removeClass('icp-loading');
                    
                    // Handle API key missing error specifically
                    if (response.data && response.data.error_type === 'missing_api_key') {
                        showApiKeyError(response.data.message, response.data.settings_url, 'single');
                    } else {
                        showNotification('Error: ' + (response.data.message || response.data || 'Unknown error'), 'error');
                    }
                }
            },
            error: function() {
                $link.text(originalText).removeClass('icp-loading');
                alert('Compression failed. Please try again.');
            }
        });
    });
    
    // API Key Testing
    $('#icp-test-api-key').on('click', function() {
        const $button = $(this);
        const $result = $('#icp-api-test-result');
        const apiKey = $('input[name="icp_settings[api_key]"]').val();
        
        if (!apiKey) {
            $result.text('Please enter an API key first.').removeClass('success').addClass('error');
            return;
        }
        
        $button.text('Testing...').prop('disabled', true);
        $result.text('').removeClass('success error');
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_test_api_key',
                nonce: icp_ajax.test_api_nonce,
                api_key: apiKey
            },
            success: function(response) {
                $button.text('Test API Key').prop('disabled', false);
                
                if (response.success) {
                    $result.text('✓ ' + response.message).addClass('success');
                } else {
                    $result.text('✗ ' + response.error).addClass('error');
                }
            },
            error: function() {
                $button.text('Test API Key').prop('disabled', false);
                $result.text('✗ Test failed. Please try again.').addClass('error');
            }
        });
    });
    
    // Auto-test API key when input changes (with delay)
    let apiKeyTestTimeout;
    $('input[name="icp_settings[api_key]"]').on('input', function() {
        const $result = $('#icp-api-test-result');
        $result.text('').removeClass('success error');
        
        clearTimeout(apiKeyTestTimeout);
        
        const apiKey = $(this).val();
        if (apiKey.length > 10) { // Basic length check
            apiKeyTestTimeout = setTimeout(function() {
                $('#icp-test-api-key').trigger('click');
            }, 2000); // Test after 2 seconds of no typing
        }
    });
    
    // Form validation
    $('form').on('submit', function() {
        const apiKey = $('input[name="icp_settings[api_key]"]').val();
        const autoCompress = $('input[name="icp_settings[auto_compress]"]').is(':checked');
        
        if (autoCompress && !apiKey) {
            alert('Please enter a TinyPNG API key before enabling auto-compression.');
            return false;
        }
    });
    
    // Dashboard refresh functionality
    if (window.location.href.indexOf('page=image-compression') > -1) {
        // Auto-refresh stats every 30 seconds on dashboard
        setInterval(function() {
            if (!bulkCompressionRunning) {
                // Only refresh if not doing bulk compression
                $('.icp-dashboard-cards').load(location.href + ' .icp-dashboard-cards > *');
            }
        }, 30000);
    }
    
    // Search and Filter functionality - Manual filtering only
    // Removed auto-submit functionality - users must click Filter button
    
    // Date range validation
    $('#date_from, #date_to').on('change', function() {
        const dateFrom = $('#date_from').val();
        const dateTo = $('#date_to').val();
        
        if (dateFrom && dateTo && dateFrom > dateTo) {
            alert('The "From" date cannot be later than the "To" date.');
            $(this).val('');
            return;
        }
        
        // No auto-submit - user must click Filter button
    });
    
    // Clear search button
    $('.icp-clear-search').on('click', function(e) {
        e.preventDefault();
        window.location.href = $(this).attr('href');
    });
    
    // Loading state for search form
    $('.icp-search-form').on('submit', function() {
        const $form = $(this);
        const $submitBtn = $form.find('input[type="submit"]');
        
        $submitBtn.val('Searching...').prop('disabled', true);
        $form.addClass('icp-loading');
        
        // Show loading indicator
        if (!$form.find('.icp-search-loading').length) {
            $form.append('<div class="icp-search-loading">Searching...</div>');
        }
    });
    
    // Keyboard shortcuts
    $(document).on('keydown', function(e) {
        // Ctrl/Cmd + K to focus search
        if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
            e.preventDefault();
            $('#search').focus();
        }
        
        // Enter key in search field
        if (e.key === 'Enter' && $('#search').is(':focus')) {
            e.preventDefault();
            $('.icp-search-form').submit();
        }
    });
    
    // Highlight search terms in results
    if ($('#search').val()) {
        const searchTerm = $('#search').val().toLowerCase();
        $('.wp-list-table tbody td').each(function() {
            const $td = $(this);
            const text = $td.text();
            const regex = new RegExp('(' + searchTerm.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + ')', 'gi');
            
            if (regex.test(text)) {
                const highlightedText = text.replace(regex, '<mark>$1</mark>');
                $td.html(highlightedText);
            }
        });
    }
    
    // Tooltips for error messages
    $(document).on('mouseenter', '.icp-status-error, .icp-compression-failed', function() {
        const title = $(this).attr('title');
        if (title) {
            $(this).attr('data-tooltip', title);
        }
    });
    
    // Loading states
    function showLoading(element) {
        $(element).addClass('icp-loading');
    }
    
    function hideLoading(element) {
        $(element).removeClass('icp-loading');
    }
    
    // Utility functions
    function formatBytes(bytes) {
        if (bytes >= 1048576) {
            return (bytes / 1048576).toFixed(2) + ' MB';
        } else if (bytes >= 1024) {
            return (bytes / 1024).toFixed(2) + ' KB';
        } else {
            return bytes + ' bytes';
        }
    }
    
    // Initialize tooltips if using a tooltip library
    if (typeof $.fn.tooltip !== 'undefined') {
        $('[data-tooltip]').tooltip();
    }
    
    // Reset Statistics functionality
    window.confirmResetStats = function() {
        if (confirm('Are you sure you want to reset all compression statistics? This action cannot be undone.')) {
            resetStats();
        }
    };
    
    // Cleanup Orphaned Records functionality
    window.confirmCleanupOrphaned = function() {
        if (confirm('This will remove compression records for images that have been deleted from your media library. Continue?')) {
            cleanupOrphanedRecords();
        }
    };

    
    function resetStats() {
        const $button = $('#icp-reset-stats');
        const originalText = $button.text();
        
        $button.text('Resetting...').prop('disabled', true);
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_reset_stats',
                nonce: icp_ajax.reset_stats_nonce
            },
            success: function(response) {
                if (response.success) {
                    alert('Statistics reset successfully!');
                    location.reload(); // Refresh the page to show updated stats
                } else {
                    alert('Error resetting statistics: ' + response.data);
                    $button.text(originalText).prop('disabled', false);
                }
            },
            error: function() {
                alert('Failed to reset statistics. Please try again.');
                $button.text(originalText).prop('disabled', false);
            }
        });
    }
    
    function cleanupOrphanedRecords() {
        const $button = $('#icp-cleanup-orphaned');
        const originalText = $button.text();
        
        $button.text('Cleaning up...').prop('disabled', true);
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_cleanup_orphaned',
                nonce: icp_ajax.cleanup_orphaned_nonce
            },
            success: function(response) {
                if (response.success) {
                    alert(response.data.message);
                    location.reload(); // Refresh the page to show updated stats
                } else {
                    alert('Error cleaning up orphaned records: ' + (response.data.message || response.data));
                    $button.text(originalText).prop('disabled', false);
                }
            },
            error: function() {
                alert('Failed to clean up orphaned records. Please try again.');
                $button.text(originalText).prop('disabled', false);
            }
        });
    }

    // Bulk Compression Settings Functionality
    $('#icp-save-bulk-folder-settings').on('click', function(e) {
        e.preventDefault();

        const $button = $(this);
        const $status = $('#icp-bulk-folder-settings-status');
        const $form = $('#icp-bulk-folder-exclusion-form');

        // Disable button and show loading
        $button.prop('disabled', true).text('Saving...');
        $status.html('<span style="color: #0073aa;">Saving settings...</span>');

        // Get form data
        const formData = new FormData($form[0]);
        formData.append('action', 'icp_save_bulk_settings');
        formData.append('nonce', icp_ajax.bulk_settings_nonce);

        // Convert FormData to regular object for jQuery
        const data = {};
        for (let [key, value] of formData.entries()) {
            if (data[key]) {
                // Handle multiple values (checkboxes)
                if (Array.isArray(data[key])) {
                    data[key].push(value);
                } else {
                    data[key] = [data[key], value];
                }
            } else {
                data[key] = value;
            }
        }

        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: data,
            success: function(response) {
                if (response.success) {
                    $status.html('<span style="color: #46b450;">✓ Settings saved successfully!</span>');

                    // Show success message and suggest page refresh
                    setTimeout(function() {
                        if (confirm('Folder exclusion settings saved successfully!\n\nWould you like to refresh the page to see updated image counts?')) {
                            location.reload();
                        }
                    }, 1000);
                } else {
                    $status.html('<span style="color: #dc3232;">✗ Error: ' + (response.data || 'Unknown error') + '</span>');
                }
            },
            error: function(xhr, status, error) {
                console.error('Bulk settings save error:', xhr, status, error);
                $status.html('<span style="color: #dc3232;">✗ Network error occurred. Please try again.</span>');
            },
            complete: function() {
                // Re-enable button
                $button.prop('disabled', false).text('Save Folder Exclusion Settings');

                // Clear status after 5 seconds
                setTimeout(function() {
                    $status.html('');
                }, 5000);
            }
        });
    });

    // Modal Compression functionality
    $(document).on('click', '.icp-compress-single-modal', function(e) {
        e.preventDefault();
        console.log('ICP: Modal compress button clicked');
        
        const $button = $(this);
        const attachmentId = $button.data('attachment-id');
        const $status = $button.closest('.icp-modal-compression-status');
        const $progress = $status.find('.icp-compression-progress');
        
        console.log('ICP: Attachment ID:', attachmentId);
        console.log('ICP: Ajax object:', typeof icp_ajax !== 'undefined' ? icp_ajax : 'undefined');
        
        if (!attachmentId) {
            alert('Attachment ID not found');
            return;
        }
        
        if (typeof icp_ajax === 'undefined') {
            alert('ICP Ajax object not found. Please refresh the page and try again.');
            return;
        }
        
        // Show progress and disable button
        $button.hide();
        $progress.show();
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_compress_single',
                nonce: icp_ajax.compress_single_nonce,
                attachment_id: attachmentId
            },
            success: function(response) {
                $progress.hide();
                
                if (response.success) {
                    // Show success message with compression details
                    let successHtml = '<span class="icp-compressed">✓ Compressed - ' + response.data.savings_percent + '% saved</span><br>';
                    successHtml += '<small>Original: ' + response.data.original_size + ' | ';
                    successHtml += 'Compressed: ' + response.data.compressed_size + ' | ';
                    successHtml += 'Saved: ' + response.data.savings_bytes + '</small>';
                    
                    $status.html(successHtml);
                    
                    // Show a success notification and refresh page after user acknowledges
                    if (typeof wp !== 'undefined' && wp.data && wp.data.select) {
                        // For WordPress 5.3+ with notices
                        const notice = wp.data.dispatch('core/notices').createNotice(
                            'success',
                            'Image compressed successfully! Saved ' + response.data.savings_percent + '%',
                            { isDismissible: true }
                        );
                        // Refresh page after a short delay to update metadata
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        // Fallback notification - refresh page after user clicks OK
                        alert('Image compressed successfully!\nSavings: ' + response.data.savings_percent + '%');
                        location.reload();
                    }
                } else {
                    // Handle API key missing error specifically
                    if (response.data && response.data.error_type === 'missing_api_key') {
                        let errorHtml = '<div class="icp-modal-api-key-error">';
                        errorHtml += '<span class="icp-status-error">🔑 API Key Required</span><br>';
                        errorHtml += '<small>' + response.data.message + '</small><br>';
                        errorHtml += '<div class="icp-modal-api-key-actions">';
                        errorHtml += '<a href="' + response.data.settings_url + '" class="button button-primary button-small">Configure API Key</a>';
                        errorHtml += '</div>';
                        errorHtml += '</div>';
                        
                        $status.html(errorHtml);
                    } else {
                        // Show error and restore button
                        let errorHtml = '<span class="icp-status-error">✗ Compression failed</span><br>';
                        errorHtml += '<small>' + (response.data.message || response.data || 'Unknown error') + '</small><br>';
                        errorHtml += '<button type="button" class="button button-secondary icp-compress-single-modal" data-attachment-id="' + attachmentId + '">Retry Compression</button>';
                        
                        $status.html(errorHtml);
                        
                        alert('Compression failed: ' + (response.data.message || response.data || 'Unknown error'));
                    }
                }
            },
            error: function(xhr, status, error) {
                $progress.hide();
                
                // Show error and restore button
                let errorHtml = '<span class="icp-status-error">✗ Compression failed</span><br>';
                errorHtml += '<small>Network error: ' + error + '</small><br>';
                errorHtml += '<button type="button" class="button button-secondary icp-compress-single-modal" data-attachment-id="' + attachmentId + '">Retry Compression</button>';
                
                $status.html(errorHtml);
                
                alert('Network error occurred: ' + error);
            },
            timeout: 60000 // 60 second timeout for single image compression
        });
    });
    
    // Auto-upload compression popup notification system
    function checkForRecentCompressions() {
        if (typeof icp_ajax === 'undefined' || !icp_ajax.get_recent_compressions_nonce) {
            return;
        }
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_get_recent_compressions',
                nonce: icp_ajax.get_recent_compressions_nonce
            },
            success: function(response) {
                if (response.success && response.data.compressions && response.data.compressions.length > 0) {
                    showCompressionPopups(response.data.compressions);
                }
            },
            error: function() {
                // Silently fail - don't show error for this background check
            }
        });
    }
    
    function showCompressionPopups(compressions) {
        compressions.forEach(function(compression, index) {
            // Show each popup with a slight delay to avoid overwhelming the user
            setTimeout(function() {
                showSingleCompressionPopup(compression);
            }, index * 1000); // 1 second delay between each popup
        });
    }
    
    function showSingleCompressionPopup(compression) {
        let popupHtml = '';
        
        if (compression.type === 'compression_success') {
            popupHtml = `
                <div class="icp-compression-popup" id="icp-popup-${compression.attachment_id}">
                    <div class="icp-popup-content">
                        <div class="icp-popup-header">
                            <h3>🎉 Image Compressed Successfully!</h3>
                            <button class="icp-popup-close" onclick="closeCompressionPopup(${compression.attachment_id})">&times;</button>
                        </div>
                        <div class="icp-popup-body">
                            <div class="icp-popup-image-title">
                                <strong>${compression.title}</strong>
                            </div>
                            <div class="icp-popup-savings">
                                <span class="icp-popup-percentage">${compression.savings_percent}% saved</span>
                            </div>
                            <div class="icp-popup-details">
                                <div class="icp-popup-detail-item">
                                    <span class="icp-popup-label">Original:</span>
                                    <span class="icp-popup-value">${compression.original_size}</span>
                                </div>
                                <div class="icp-popup-detail-item">
                                    <span class="icp-popup-label">Compressed:</span>
                                    <span class="icp-popup-value">${compression.compressed_size}</span>
                                </div>
                                <div class="icp-popup-detail-item">
                                    <span class="icp-popup-label">Saved:</span>
                                    <span class="icp-popup-value">${compression.savings_bytes}</span>
                                </div>
                            </div>
                        </div>
                        <div class="icp-popup-footer">
                            <button class="button button-primary" onclick="closeCompressionPopup(${compression.attachment_id})">Got it!</button>
                        </div>
                    </div>
                </div>
            `;
        } else if (compression.type === 'missing_api_key') {
            popupHtml = `
                <div class="icp-compression-popup icp-popup-warning" id="icp-popup-${compression.attachment_id}">
                    <div class="icp-popup-content">
                        <div class="icp-popup-header">
                            <h3>🔑 Image Uploaded - Compression Available</h3>
                            <button class="icp-popup-close" onclick="closeCompressionPopup(${compression.attachment_id})">&times;</button>
                        </div>
                        <div class="icp-popup-body">
                            <div class="icp-popup-image-title">
                                <strong>${compression.title}</strong>
                            </div>
                            <div class="icp-popup-message">
                                <p>Your image has been uploaded successfully!</p>
                                <p>💾 <strong>Save space and improve performance</strong> by enabling automatic image compression.</p>
                                <div class="icp-popup-file-info">
                                    <span class="icp-popup-label">File size:</span>
                                    <span class="icp-popup-value">${compression.file_size}</span>
                                </div>
                            </div>
                        </div>
                        <div class="icp-popup-footer">
                            <a href="${compression.settings_url}" class="button button-primary">Enter TinyPNG API Key</a>
                            <button class="button button-secondary" onclick="closeCompressionPopup(${compression.attachment_id})">Maybe Later</button>
                        </div>
                    </div>
                </div>
            `;
        } else if (compression.type === 'compression_failed') {
            popupHtml = `
                <div class="icp-compression-popup icp-popup-error" id="icp-popup-${compression.attachment_id}">
                    <div class="icp-popup-content">
                        <div class="icp-popup-header">
                            <h3>⚠️ Image Uploaded - Compression Failed</h3>
                            <button class="icp-popup-close" onclick="closeCompressionPopup(${compression.attachment_id})">&times;</button>
                        </div>
                        <div class="icp-popup-body">
                            <div class="icp-popup-image-title">
                                <strong>${compression.title}</strong>
                            </div>
                            <div class="icp-popup-message">
                                <p>✅ <strong>Your image has been uploaded successfully!</strong></p>
                                <p>❌ <strong>However, image compression failed due to the following reason:</strong></p>
                                <div class="icp-popup-error-details">
                                    <p><strong>Error:</strong> ${compression.error_message}</p>
                                </div>
                                <div class="icp-popup-file-info">
                                    <span class="icp-popup-label">File size:</span>
                                    <span class="icp-popup-value">${compression.file_size}</span>
                                </div>
                            </div>
                        </div>
                        <div class="icp-popup-footer">
                            <button class="button button-primary" onclick="retryCompressionFromPopup(${compression.attachment_id})">Retry Compression</button>
                            <button class="button button-secondary" onclick="closeCompressionPopup(${compression.attachment_id})">OK</button>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // Add popup to body
        $('body').append(popupHtml);
        
        // Show popup with animation
        const popup = $(`#icp-popup-${compression.attachment_id}`);
        setTimeout(function() {
            popup.addClass('icp-popup-visible');
        }, 50);
        
        // Auto-close after 10 seconds for success, 15 seconds for API key warning, 20 seconds for compression failure
        let autoCloseDelay = 10000; // Default for success
        if (compression.type === 'missing_api_key') {
            autoCloseDelay = 15000;
        } else if (compression.type === 'compression_failed') {
            autoCloseDelay = 20000; // Longer delay for error messages
        }
        setTimeout(function() {
            closeCompressionPopup(compression.attachment_id);
        }, autoCloseDelay);
    }
    
    // Global function to close popup (called from inline onclick)
    window.closeCompressionPopup = function(attachmentId) {
        const popup = $(`#icp-popup-${attachmentId}`);
        popup.removeClass('icp-popup-visible');
        setTimeout(function() {
            popup.remove();
        }, 300);
    };
    
    // Global function to retry compression from popup (called from inline onclick)
    window.retryCompressionFromPopup = function(attachmentId) {
        // Close the current popup
        closeCompressionPopup(attachmentId);
        
        // Show loading notification
        showNotification('Retrying compression...', 'info');
        
        // Perform the compression retry
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_compress_single',
                nonce: icp_ajax.compress_single_nonce,
                attachment_id: attachmentId
            },
            success: function(response) {
                if (response.success) {
                    showNotification('✅ Image compressed successfully! Saved ' + response.data.savings_percent + '%', 'success');
                    
                    // Refresh the page to update the media library
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    let errorMessage = 'Compression failed';
                    if (response.data && response.data.message) {
                        errorMessage += ': ' + response.data.message;
                    } else if (response.data) {
                        errorMessage += ': ' + response.data;
                    }
                    showNotification('❌ ' + errorMessage, 'error');
                }
            },
            error: function(xhr, status, error) {
                showNotification('❌ Network error occurred: ' + error, 'error');
            },
            timeout: 60000
        });
    };
    
    // Check for recent compressions on page load and periodically
    $(document).ready(function() {
        // Check immediately on page load
        setTimeout(checkForRecentCompressions, 1000);
        
        // Check every 5 seconds for new compressions (useful for multi-upload scenarios)
        setInterval(checkForRecentCompressions, 5000);
    });
    
    // ==========================================
    // SCHEDULER FUNCTIONALITY
    // ==========================================
    
    // Handle batch mode radio button changes
    $('input[name="icp-batch-mode"]').on('change', function() {
        if ($(this).val() === 'unlimited') {
            $('#icp-batch-size-container').hide();
        } else {
            $('#icp-batch-size-container').show();
        }
    });
    
    // Update scheduler settings
    $('#icp-update-scheduler').on('click', function() {
        const $button = $(this);
        const enabled = $('#icp-scheduler-enabled').is(':checked');
        const interval = $('#icp-scheduler-interval').val();
        
        // Determine batch size based on selected mode
        let batchSize;
        if ($('#icp-batch-unlimited').is(':checked')) {
            batchSize = -1; // Unlimited
        } else {
            batchSize = $('#icp-scheduler-batch-size').val();
        }
        
        // Get excluded folders
        const excludedFolders = [];
        $('input[name="excluded_folders[]"]:checked').each(function() {
            excludedFolders.push($(this).val());
        });
        
        $button.prop('disabled', true).text('Saving...');
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_update_schedule',
                nonce: icp_ajax.scheduler_nonce,
                enabled: enabled,
                interval: interval,
                batch_size: batchSize,
                excluded_folders: excludedFolders
            },
            success: function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    // Refresh the page to show updated status
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification('Failed to update scheduler settings: ' + (response.data.message || response.data), 'error');
                }
            },
            error: function(xhr, status, error) {
                showNotification('Network error occurred: ' + error, 'error');
            },
            complete: function() {
                $button.prop('disabled', false).text('Save Settings');
            }
        });
    });
    
    // Run scheduled compression manually
    $('#icp-run-scheduler-now').on('click', function() {
        const $button = $(this);
        
        $button.prop('disabled', true).text('Running...');
        $('#icp-scheduler-status').html('<span class="spinner is-active" style="float:none;margin:0 5px;"></span>Running compression...');
        
        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_run_scheduled_compression_now',
                nonce: icp_ajax.scheduler_nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    $('#icp-scheduler-status').html('<span style="color: green;">✓ Compression run completed</span>');
                    
                    // Refresh the page to show updated status and logs
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification('Failed to run scheduled compression: ' + (response.data.message || response.data), 'error');
                    $('#icp-scheduler-status').html('<span style="color: red;">✗ Compression run failed</span>');
                }
            },
            error: function(xhr, status, error) {
                showNotification('Network error occurred: ' + error, 'error');
                $('#icp-scheduler-status').html('<span style="color: red;">✗ Network error</span>');
            },
            complete: function() {
                $button.prop('disabled', false).text('Run Now');
                // Clear status message after a few seconds
                setTimeout(function() {
                    $('#icp-scheduler-status').empty();
                }, 5000);
            }
        });
    });
    
    // Auto-refresh scheduler status every 30 seconds if on scheduler page
    if (window.location.href.indexOf('image-compression-scheduler') !== -1) {
        setInterval(function() {
            $.ajax({
                url: icp_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'icp_get_schedule_status',
                    nonce: icp_ajax.scheduler_nonce
                },
                success: function(response) {
                    if (response.success && response.data.status) {
                        const status = response.data.status;
                        
                        // Update status indicators
                        const statusIndicator = $('.icp-status-indicator');
                        if (status.enabled) {
                            statusIndicator.removeClass('disabled').addClass('enabled').text('● Active');
                        } else {
                            statusIndicator.removeClass('enabled').addClass('disabled').text('○ Inactive');
                        }
                        
                        // Update next run time
                        if (status.next_run) {
                            $('.icp-next-run').text(status.next_run);
                            $('.icp-no-schedule').hide();
                        } else {
                            $('.icp-next-run').hide();
                            $('.icp-no-schedule').show();
                        }
                        
                        // Update last run information
                        if (status.last_run) {
                            $('.icp-last-run').text(status.last_run);
                            if (status.last_run_stats) {
                                const stats = status.last_run_stats;
                                let statsText = stats.successful + '/' + stats.processed + ' images compressed';
                                if (stats.errors > 0) {
                                    statsText += ' | ' + stats.errors + ' errors';
                                }
                                $('.icp-last-run-stats small').text(statsText);
                            }
                            $('.icp-no-run').hide();
                        } else {
                            $('.icp-last-run').hide();
                            $('.icp-last-run-stats').hide();
                            $('.icp-no-run').show();
                        }
                    }
                },
                error: function() {
                    // Silently fail - don't show errors for background status updates
                }
            });
        }, 30000); // Every 30 seconds
    }

    // ==========================================
    // RESTORE FUNCTIONALITY
    // ==========================================

    // Handle single image restore
    $(document).on('click', '.icp-restore-single', function(e) {
        e.preventDefault();

        const $button = $(this);
        const attachmentId = $button.data('attachment-id');

        if (!attachmentId) {
            showNotification('❌ Invalid attachment ID', 'error');
            return;
        }

        // Confirm restore action
        if (!confirm('Are you sure you want to restore the original image? This will replace the compressed version.')) {
            return;
        }

        const originalText = $button.text();
        $button.prop('disabled', true).text('Restoring...');

        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_restore_single',
                nonce: icp_ajax.restore_single_nonce,
                attachment_id: attachmentId
            },
            success: function(response) {
                if (response.success) {
                    showNotification('✅ Original image restored successfully!', 'success');

                    // Refresh the page to update the interface
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    let errorMessage = 'Restore failed';
                    if (response.data && response.data.error) {
                        errorMessage += ': ' + response.data.error;
                    } else if (response.data) {
                        errorMessage += ': ' + response.data;
                    }
                    showNotification('❌ ' + errorMessage, 'error');
                    $button.prop('disabled', false).text(originalText);
                }
            },
            error: function(xhr, status, error) {
                showNotification('❌ Network error: ' + error, 'error');
                $button.prop('disabled', false).text(originalText);
            }
        });
    });

    // Handle bulk restore form submission
    $('#icp-bulk-restore-form').on('submit', function(e) {
        e.preventDefault();

        const action = $('select[name="action"]').val() || $('select[name="action2"]').val();

        if (action !== 'restore') {
            return;
        }

        const selectedIds = [];
        $('input[name="attachment_ids[]"]:checked').each(function() {
            selectedIds.push($(this).val());
        });

        if (selectedIds.length === 0) {
            showNotification('❌ Please select at least one image to restore', 'error');
            return;
        }

        // Confirm bulk restore action
        const confirmMessage = `Are you sure you want to restore ${selectedIds.length} original image(s)? This will replace the compressed versions.`;
        if (!confirm(confirmMessage)) {
            return;
        }

        const $submitButtons = $('#doaction, #doaction2');
        $submitButtons.prop('disabled', true).val('Restoring...');

        // Show progress indicator
        const $progressDiv = $('<div id="icp-restore-progress" style="margin: 20px 0; padding: 15px; background: #f1f1f1; border-left: 4px solid #0073aa;"></div>');
        $progressDiv.html('<span class="spinner is-active" style="float:none;margin:0 5px;"></span>Restoring images, please wait...');
        $(this).before($progressDiv);

        $.ajax({
            url: icp_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'icp_bulk_restore',
                nonce: icp_ajax.bulk_restore_nonce,
                attachment_ids: selectedIds
            },
            success: function(response) {
                if (response.success) {
                    const results = response.data;
                    let message = `✅ Bulk restore completed: ${results.success} restored`;

                    if (results.failed > 0) {
                        message += `, ${results.failed} failed`;
                    }

                    showNotification(message, results.failed > 0 ? 'warning' : 'success');

                    // Show detailed error information if any
                    if (results.errors && results.errors.length > 0) {
                        let errorDetails = 'Failed restorations:\n';
                        results.errors.forEach(function(error) {
                            errorDetails += `- Attachment ${error.attachment_id}: ${error.error}\n`;
                        });
                        console.log(errorDetails);
                    }

                    // Refresh the page to show updated results
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    let errorMessage = 'Bulk restore failed';
                    if (response.data) {
                        errorMessage += ': ' + response.data;
                    }
                    showNotification('❌ ' + errorMessage, 'error');
                }
            },
            error: function(xhr, status, error) {
                showNotification('❌ Network error during bulk restore: ' + error, 'error');
            },
            complete: function() {
                $submitButtons.prop('disabled', false).val('Apply');
                $('#icp-restore-progress').remove();
            }
        });
    });

    // Handle select all checkbox for restore page
    $('#cb-select-all-1').on('change', function() {
        const isChecked = $(this).is(':checked');
        $('input[name="attachment_ids[]"]').prop('checked', isChecked);
    });

    // Update select all checkbox when individual checkboxes change
    $(document).on('change', 'input[name="attachment_ids[]"]', function() {
        const totalCheckboxes = $('input[name="attachment_ids[]"]').length;
        const checkedCheckboxes = $('input[name="attachment_ids[]"]:checked').length;

        $('#cb-select-all-1').prop('checked', totalCheckboxes === checkedCheckboxes);
    });

});